#ifndef _diffusion_update_constraint_value_h_
#define _diffusion_update_constraint_value_h_ 1

/*
 * Copyright © 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to licence terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 */

/**
 * @file update-constraint-value.h
 *
 * A constraint value defines a datatype and associated value to be used in
 * a topic update constraint.
 *
 * @since 6.10
 */

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>

#include <math.h>
#include <time.h>

/**
 * Opaque topic update constraint value struct
 */
typedef struct DIFFUSION_UPDATE_CONSTRAINT_VALUE_T DIFFUSION_UPDATE_CONSTRAINT_VALUE_T;


/**
 * Creates an update constraint value from a string value.
 *
 * `diffusion_update_constraint_value_free` should be called on the pointer when no longer needed.
 *
 * @param value     the string value
 *
 * @return          an update constraint value or `NULL` if an error has occurred.
 *
 * @since 6.10
 */
DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *diffusion_update_constraint_value_from_string(char *value);


/**
 * Creates an update constraint value from an Int64 value.
 *
 * `diffusion_update_constraint_value_free` should be called on the pointer when no longer needed.
 *
 * @param value     the Int64 value
 *
 * @return          an update constraint value or `NULL` if an error has occurred.
 *
 * @since 6.10
 */
DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *diffusion_update_constraint_value_from_int64(int64_t value);


/**
 * Creates an update constraint value from a double value.
 *
 * `diffusion_update_constraint_value_free` should be called on the pointer when no longer needed.
 *
 * @param value     the double value
 *
 * @return          an update constraint value or `NULL` if an error has occurred.
 *
 * @since 6.10
 */
DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *diffusion_update_constraint_value_from_double(double value);


/**
 * Creates an update constraint value from a binary value.
 *
 * `diffusion_update_constraint_value_free` should be called on the pointer when no longer needed.
 *
 * @param value     the binary value
 * @param len       the length of the binary value
 *
 * @return          an update constraint value or `NULL` if an error has occurred.
 *
 * @since 6.10
 */
DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *diffusion_update_constraint_value_from_binary(char *value, size_t len);


/**
 * Creates an update constraint value from a json string.
 *
 * `diffusion_update_constraint_value_free` should be called on the pointer when no longer needed.
 *
 * @param value     the json string
 *
 * @return          an update constraint value or `NULL` if an error has occurred.
 *
 * @since 6.10
 */
DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *diffusion_update_constraint_value_from_json_string(char *value);


/**
 * Creates an update constraint value from a recordV2 string.
 *
 * `diffusion_update_constraint_value_free` should be called on the pointer when no longer needed.
 *
 * @param value     the recordV2 string
 *
 * @return          an update constraint value or `NULL` if an error has occurred.
 *
 * @since 6.10
 */
DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *diffusion_update_constraint_value_from_recordv2_string(char *value);


/**
 * Creates an update constraint value from a boolean value.
 *
 * `diffusion_update_constraint_value_free` should be called on the pointer when no longer needed.
 *
 * @param value     the boolean value
 *
 * @return          an update constraint value or `NULL` if an error has occurred.
 *
 * @since 6.10
 */
DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *diffusion_update_constraint_value_from_bool(bool value);


/**
 * Frees an update constraint value.
 *
 * @param value     the update constraint value
 *
 * @since 6.10
 */
void diffusion_update_constraint_value_free(DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *value);


/**
 * Creates a duplicate of an update constraint value
 *
 * `diffusion_update_constraint_value_free` should be called on the pointer when no longer needed.
 *
 * @param value     the update constraint value
 *
 * @return          a duplicate of the original update constraint value
 *
 * @since 6.10
 */
DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *diffusion_update_constraint_value_dup(DIFFUSION_UPDATE_CONSTRAINT_VALUE_T *value);


#endif
