/*
 * Decompiled with CFR 0.152.
 */
package com.pushtechnology.diffusion.utils.bytebuffer;

import com.pushtechnology.diffusion.logs.i18n.I18nLogger;
import com.pushtechnology.diffusion.utils.ConfigurationUtils;
import com.pushtechnology.diffusion.utils.bytebuffer.FreeDirectByteBuffer;
import com.pushtechnology.diffusion.utils.bytebuffer.ManagedByteBufferPool;
import com.pushtechnology.diffusion.utils.math.DiffusionMath;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import net.jcip.annotations.Immutable;
import net.jcip.annotations.NotThreadSafe;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;

@NotThreadSafe
class ByteBufferPoolImpl
implements ManagedByteBufferPool {
    private static final Logger LOG = I18nLogger.getLogger(ByteBufferPoolImpl.class);
    static final Parameters BUFFER_POOL_DEFAULTS = new Parameters();
    private final Pool[] pools = new Pool[32];
    private final Parameters parameters;
    private final Account account = new Account();
    private final AtomicBoolean destroyed = new AtomicBoolean();
    static final Account GLOBAL_ACCOUNT = new Account();
    static final boolean GLOBAL_POOL_ACCOUNTING = ConfigurationUtils.getBooleanSystemProperty("diffusion.global_pool_accounting");

    ByteBufferPoolImpl(Parameters parameters) {
        this.parameters = parameters;
    }

    private ByteBuffer allocate(int capacity) {
        if (GLOBAL_POOL_ACCOUNTING) {
            GLOBAL_ACCOUNT.allocate();
        }
        this.account.allocate();
        return ByteBuffer.allocateDirect(capacity);
    }

    @Override
    public ByteBuffer provide(int minimumCapacity) {
        ByteBuffer existing = this.poll(minimumCapacity);
        if (existing != null) {
            return existing;
        }
        ByteBuffer allocated = this.allocate(ByteBufferPoolImpl.poolCapacityFor(ByteBufferPoolImpl.slotFor(this.parameters.capacityFor(minimumCapacity))));
        if (LOG.isTraceEnabled()) {
            LOG.trace("provide({}) allocated {}", (Object)minimumCapacity, (Object)allocated);
        }
        return allocated;
    }

    protected final ByteBuffer poll(int minimumCapacity) {
        Pool pool = this.ensurePool(ByteBufferPoolImpl.slotFor(this.parameters.capacityFor(minimumCapacity)));
        ByteBuffer result = pool.poll();
        if (LOG.isTraceEnabled()) {
            LOG.trace("poll({}) -> {} from {}", minimumCapacity, result, this);
        }
        return result;
    }

    protected final boolean offer(ByteBuffer buffer) {
        assert (buffer.isDirect()) : buffer;
        Pool existing = this.pools[ByteBufferPoolImpl.slotFor(buffer.capacity())];
        LOG.trace("offer({}) to {}", (Object)buffer, (Object)this);
        if (existing != null) {
            return existing.offer(buffer);
        }
        return false;
    }

    @Override
    public final void release(ByteBuffer buffer) {
        if (buffer.isDirect() && !this.offer(buffer)) {
            LOG.trace("discarded {}", (Object)buffer);
            FreeDirectByteBuffer.freeDirectBuffer(buffer);
            this.account.freed();
            if (GLOBAL_POOL_ACCOUNTING) {
                GLOBAL_ACCOUNT.freed();
            }
        }
    }

    private Pool ensurePool(int slot) {
        Pool newPool;
        Pool existing = this.pools[slot];
        if (existing != null) {
            return existing;
        }
        this.pools[slot] = newPool = new Pool(ByteBufferPoolImpl.poolCapacityFor(slot));
        return newPool;
    }

    private static int slotFor(int capacity) {
        assert (capacity > 0) : capacity;
        return DiffusionMath.highestBit(capacity - 1);
    }

    static int poolCapacityFor(int slot) {
        return slot == 31 ? Integer.MAX_VALUE : 1 << slot;
    }

    public final void destroy() {
        if (this.destroyed.compareAndSet(false, true)) {
            LOG.trace("destroying {}", (Object)this);
            for (Pool pool : this.pools) {
                if (pool == null) continue;
                pool.destroy();
            }
            LOG.trace("destroyed  {}", (Object)this);
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(256);
        sb.append(this.parameters.getName());
        sb.append('[');
        sb.append(this.account);
        for (Pool p : this.pools) {
            if (p == null) continue;
            sb.append(' ');
            sb.append(p);
        }
        sb.append(']');
        return sb.toString();
    }

    @Override
    public final ManagedByteBufferPool.Statistics statistics() {
        return this.account;
    }

    protected void finalize() throws Throwable {
        this.destroy();
        super.finalize();
    }

    @NotThreadSafe
    private final class Pool {
        private final int capacity;
        private final Deque<ByteBuffer> stack;
        private long cacheHits;
        private long cacheMisses;
        private long dropped;

        Pool(int capacity) {
            this.capacity = capacity;
            int initialBuffers = ByteBufferPoolImpl.this.parameters.getInitialPerCapacity();
            this.stack = new ArrayDeque<ByteBuffer>(initialBuffers);
            for (int i = 0; i < initialBuffers; ++i) {
                this.stack.add(ByteBufferPoolImpl.this.allocate(capacity));
            }
            LOG.trace("allocated ({})", (Object)this);
        }

        ByteBuffer poll() {
            ByteBuffer buffer = this.stack.poll();
            if (buffer != null) {
                ++this.cacheHits;
                return buffer;
            }
            ++this.cacheMisses;
            return null;
        }

        boolean offer(ByteBuffer buffer) {
            assert (this.capacity == buffer.capacity()) : buffer;
            if (this.stack.size() < ByteBufferPoolImpl.this.parameters.getMaximumPerCapacity()) {
                buffer.clear();
                this.stack.push(buffer);
                return true;
            }
            ++this.dropped;
            return false;
        }

        void destroy() {
            LOG.trace("destroy({})", (Object)this);
            ByteBuffer buffer = this.stack.poll();
            while (buffer != null) {
                FreeDirectByteBuffer.freeDirectBuffer(buffer);
                if (GLOBAL_POOL_ACCOUNTING) {
                    GLOBAL_ACCOUNT.freed();
                }
                ByteBufferPoolImpl.this.account.freed();
                buffer = this.stack.poll();
            }
        }

        public String toString() {
            StringBuilder sb = new StringBuilder(128);
            sb.append("Pool(");
            if (this.capacity < 1024) {
                sb.append(this.capacity).append(' ');
            } else {
                sb.append(this.capacity / 1024).append("k ");
            }
            sb.append(this.stack.size()).append('/').append(ByteBufferPoolImpl.this.parameters.getMaximumPerCapacity() == Integer.MAX_VALUE ? "unlimited" : Integer.toString(ByteBufferPoolImpl.this.parameters.getMaximumPerCapacity())).append(' ').append(this.cacheHits).append('/').append(this.cacheMisses).append('/').append(this.dropped).append(')');
            return sb.toString();
        }
    }

    @ThreadSafe
    static final class Account
    implements ManagedByteBufferPool.Statistics {
        private final AtomicLong allocated;
        private final AtomicLong freed;

        private void allocate() {
            this.allocated.incrementAndGet();
        }

        private void freed() {
            this.freed.incrementAndGet();
        }

        long getAllocated() {
            return this.allocated.get();
        }

        long getFreed() {
            return this.freed.get();
        }

        private Account(long alloc, long free) {
            this.allocated = new AtomicLong(alloc);
            this.freed = new AtomicLong(free);
        }

        private Account() {
            this(0L, 0L);
        }

        @Override
        public ManagedByteBufferPool.Statistics add(ManagedByteBufferPool.Statistics acc) {
            Account toadd = (Account)acc;
            return new Account(this.allocated.get() + toadd.getAllocated(), this.freed.get() + toadd.getFreed());
        }

        @Override
        public long delta() {
            return this.allocated.get() - this.freed.get();
        }

        public String toString() {
            return "+" + String.valueOf(this.allocated) + ",-" + String.valueOf(this.freed);
        }
    }

    @Immutable
    public static final class Parameters {
        private final int initialPerCapacity;
        private final int maximumPerCapacity;
        private final int minimumCapacity;
        private final String name;

        private Parameters() {
            this(0, Integer.MAX_VALUE, 1, "ByteBufferPoolImpl");
        }

        private Parameters(int initialPerCapacity, int maximumPerCapacity, int minimumCapacity, String name) {
            this.initialPerCapacity = initialPerCapacity;
            this.maximumPerCapacity = maximumPerCapacity;
            this.minimumCapacity = minimumCapacity;
            this.name = name;
        }

        public Parameters minimumCapacity(int c) {
            return new Parameters(this.initialPerCapacity, this.maximumPerCapacity, Math.max(1, c), this.name);
        }

        public Parameters initialPerCapacity(int c) {
            return new Parameters(c, this.maximumPerCapacity, this.minimumCapacity, this.name);
        }

        public Parameters maximumPerCapacity(int c) {
            return new Parameters(this.initialPerCapacity, c, this.minimumCapacity, this.name);
        }

        public Parameters name(String myname) {
            return new Parameters(this.initialPerCapacity, this.maximumPerCapacity, this.minimumCapacity, myname);
        }

        int getInitialPerCapacity() {
            return this.initialPerCapacity;
        }

        int getMaximumPerCapacity() {
            return this.maximumPerCapacity;
        }

        String getName() {
            return this.name;
        }

        int capacityFor(int minimumRequested) {
            return Math.max(minimumRequested, this.minimumCapacity);
        }
    }
}

