/*******************************************************************************
 * Copyright (c) 2022, 2025 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of DiffusionData. The intellectual and technical
 * concepts contained herein are proprietary to DiffusionData and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.diffusiondata.gateway.framework.converters;

import com.diffusiondata.gateway.framework.TopicType;
import com.diffusiondata.gateway.framework.exceptions.PayloadConversionException;
import com.pushtechnology.diffusion.client.Diffusion;
import com.pushtechnology.diffusion.datatype.DataTypes;
import com.pushtechnology.diffusion.datatype.binary.BinaryDataType;
import com.pushtechnology.diffusion.datatype.json.JSON;
import com.pushtechnology.diffusion.datatype.json.JSONDataType;

/**
 * The base interface that a payload converter implements.
 * <p>
 * For full details about payload converters see the {@link
 * com.diffusiondata.gateway.framework.converters Package Guide}.
 *
 * @param <I> the input value type to convert from
 * @param <O> the output value type to convert to.
 * @author DiffusionData Limited
 */
public interface PayloadConverter<I, O> {
    /**
     * Diffusion datatype implementation that can be used to create Diffusion
     * topic types.
     */
    DataTypes DATA_TYPES = Diffusion.dataTypes();

    /**
     * JSON datatype implementation.
     */
    JSONDataType JSON_DATA_TYPE = DATA_TYPES.json();

    /**
     * Binary datatype implementation.
     */
    BinaryDataType BINARY_DATA_TYPE = DATA_TYPES.binary();

    /**
     * The JSON 'null' value.
     */
    JSON JSON_NULL = JSON_DATA_TYPE.fromJsonString("null");

    /**
     * The name of payload converter that converts an Object to the Diffusion
     * {@link TopicType#JSON} topic format.
     * <p>
     * Supported input values include basic POJOs, primitives, and collections
     * such as Map and List.
     */
    String OBJECT_TO_JSON = "$Object_to_JSON";

    /**
     * The name of payload converter that converts the Diffusion {@link
     * TopicType#JSON} topic format to a String.
     * <p>
     * see the {@link com.diffusiondata.gateway.framework.converters Package
     * Guide} for full details of the conversions this converter performs.
     */
    String JSON_TO_STRING = "$JSON_to_String";

    /**
     * The name of payload converter that converts an Object to a String.
     * <p>
     * This can be used to publish to the Diffusion {@link TopicType#STRING}
     * topic type.
     * <p>
     * see the {@link com.diffusiondata.gateway.framework.converters Package
     * Guide} for full details of the conversions this converter performs.
     */
    String OBJECT_TO_STRING = "$Object_to_String";

    /**
     * The name of payload converter that converts an Object to a Long.
     * <p>
     * This can be used to publish to the Diffusion {@link TopicType#INT64}
     * topic type.
     * <p>
     * see the {@link com.diffusiondata.gateway.framework.converters Package
     * Guide} for full details of the conversions this converter performs.
     */
    String OBJECT_TO_LONG = "$Object_to_Long";

    /**
     * The name of payload converter that converts an Object to a Double.
     * <p>
     * This can be used to publish to the Diffusion {@link TopicType#DOUBLE}
     * topic type.
     * <p>
     * see the {@link com.diffusiondata.gateway.framework.converters Package
     * Guide} for full details of the conversions this converter performs.
     */
    String OBJECT_TO_DOUBLE = "$Object_to_Double";

    /**
     * The name of payload converter that converts an Object to the Diffusion
     * {@link TopicType#BINARY} topic type.
     * <p>
     * see the {@link com.diffusiondata.gateway.framework.converters Package
     * Guide} for full details of the conversions this converter performs.
     */
    String OBJECT_TO_BINARY_TOPIC = "$Object_to_BINARY";

    /**
     * The name of payload converter that converts the Diffusion {@link
     * TopicType#BINARY} topic type to a byte array.
     * <p>
     * see the {@link com.diffusiondata.gateway.framework.converters Package
     * Guide} for full details of the conversions this converter performs.
     */
    String BINARY_TOPIC_TO_BYTE_ARRAY = "$BINARY_to_byte_array";

    /**
     * The name of payload converter that converts an Avro value to the
     * Diffusion {@link TopicType#JSON} topic format.
     */
    String AVRO_TO_JSON = "$Avro_to_JSON";

    /**
     * The name of payload converter that converts an Avro value to a JSON
     * String.
     */
    String AVRO_TO_JSON_STRING = "$Avro_to_JSON_String";

    /**
     * The name of payload converter that converts an Avro value to an Avro byte
     * array.
     */
    String AVRO_TO_AVRO_BYTES = "$Avro_to_Avro_bytes";

    /**
     * The name of payload converter that converts an Avro byte array to the
     * Diffusion JSON topic format.
     */
    String AVRO_BYTES_TO_JSON = "$Avro_bytes_to_JSON";

    /**
     * The name of payload converter that converts the Diffusion {@link
     * TopicType#JSON} topic format to an Avro format.
     */
    String JSON_TO_AVRO = "$JSON_to_Avro";

    /**
     * The name of payload converter that converts CSV data to the Diffusion
     * {@link TopicType#JSON} topic format.
     */
    String CSV_TO_JSON = "$CSV_to_JSON";

    /**
     * The name of this payload converter that converts the Diffusion {@link
     * TopicType#JSON} topic format to CSV string.
     */
    String JSON_TO_CSV = "$JSON_to_CSV_String";

    /**
     * Converts an input value of type 'I' to an output value of type 'O'.
     *
     * @param input the unconverted input value
     * @return converted output value
     * @throws PayloadConversionException if the converter fails to convert the
     *                                    value
     */
    O convert(I input) throws PayloadConversionException;
}
